﻿using System;
using static Spectral1_VBClassLibrary.DataSet_Spectral;
using static Spectral1.BUSINESS_LOGIC.BL_MIDI;
using Spectral1.BUSINESS_LOGIC;

namespace Spectral1.DATA_ACCESS
{
    public class DA_Spectral
    {
        #region "==================== DECLARATIONS ==========================="
        public const Int32 dataset_schema_version = 0;

        public const Int32 max_harmonics = 32;
        public const Int32 max_body_resonance_filter_bands = 32;
        public const Int32 max_lfos = 3;
        public const Int32 max_oscillators = 2; //CHANGED 120121 !!!
        public const Int32 max_adsr_sections = 4;
        public const Int32 max_biquad_filters = 2;
        public const Int32 max_envelopes = 10;
        public const Int32 max_waveforms = 5;
        public const Int32 max_intensity_layers = 3;
        public const Int32 max_note_sectors = 5;
        public const Int32 max_hamonic_level_value = 65535;

        public const Int32 min_note_number = 21;
        public const Int32 max_note_number = 108;
        public const Int32 max_midi_channels = 6;
        public const Int32 max_tone_processors = 6;
        public const Int32 patches_per_patchset = 18;
        public const Int32 performances_per_patchset = 8;
        public const Int32 module_patch_write_to_EEPROM_duration_ms = 500;
        public const Int32 module_pause_to_process_ms = 10;
        public const Int32 lfo_event_period = 10;

        public const UInt16 wavetable_samples = 2048;
        public const Double sample_rate = 41666;
        public const Double A0FrequencyHz = 27.5;
        public const Int32 samples_per_ms = 42;//41.666 but needs to be an integer for envelope dispay;
        public const Int32 timbre_update_event_period = 10;
        public const Int32 c4_note_number = 60;
        public const Int32 a0_note_number = 21;
        public const Int32 c8_note_number = 108;
        public const Int32 notes_per_octave = 12;

        public const Int32 max_regular_and_random_detuning = 50;
        public const Int32 max_oscillator_detuning = 100;

        public const Int32 max_haas_delay = 25;
        public const Single detune_linear_value_per_hz = (Single)1.572864;

        public c_data_changed_overview data_changed;

        #endregion

        #region "======================= SINE TABLE======================================"
        public static readonly Int16[] SineLookup = {
        0,  101,    201,    302,    402,    503,    603,    704,
        804,    905,    1005,   1106,   1206,   1307,   1407,   1507,
        1608,   1708,   1809,   1909,   2009,   2110,   2210,   2310,
        2410,   2511,   2611,   2711,   2811,   2911,   3012,   3112,
        3212,   3312,   3412,   3512,   3612,   3712,   3811,   3911,
        4011,   4111,   4210,   4310,   4410,   4509,   4609,   4708,
        4808,   4907,   5007,   5106,   5205,   5305,   5404,   5503,
        5602,   5701,   5800,   5899,   5998,   6096,   6195,   6294,
        6393,   6491,   6590,   6688,   6786,   6885,   6983,   7081,
        7179,   7277,   7375,   7473,   7571,   7669,   7767,   7864,
        7962,   8059,   8157,   8254,   8351,   8448,   8545,   8642,
        8739,   8836,   8933,   9030,   9126,   9223,   9319,   9416,
        9512,   9608,   9704,   9800,   9896,   9992,   10087,  10183,
        10278,  10374,  10469,  10564,  10659,  10754,  10849,  10944,
        11039,  11133,  11228,  11322,  11417,  11511,  11605,  11699,
        11793,  11886,  11980,  12074,  12167,  12260,  12353,  12446,
        12539,  12632,  12725,  12817,  12910,  13002,  13094,  13187,
        13279,  13370,  13462,  13554,  13645,  13736,  13828,  13919,
        14010,  14101,  14191,  14282,  14372,  14462,  14553,  14643,
        14732,  14822,  14912,  15001,  15090,  15180,  15269,  15358,
        15446,  15535,  15623,  15712,  15800,  15888,  15976,  16063,
        16151,  16238,  16325,  16413,  16499,  16586,  16673,  16759,
        16846,  16932,  17018,  17104,  17189,  17275,  17360,  17445,
        17530,  17615,  17700,  17784,  17869,  17953,  18037,  18121,
        18204,  18288,  18371,  18454,  18537,  18620,  18703,  18785,
        18868,  18950,  19032,  19113,  19195,  19276,  19357,  19438,
        19519,  19600,  19680,  19761,  19841,  19921,  20000,  20080,
        20159,  20238,  20317,  20396,  20475,  20553,  20631,  20709,
        20787,  20865,  20942,  21019,  21096,  21173,  21250,  21326,
        21403,  21479,  21554,  21630,  21705,  21781,  21856,  21930,
        22005,  22079,  22154,  22227,  22301,  22375,  22448,  22521,
        22594,  22667,  22739,  22812,  22884,  22956,  23027,  23099,
        23170,  23241,  23311,  23382,  23452,  23522,  23592,  23662,
        23731,  23801,  23870,  23938,  24007,  24075,  24143,  24211,
        24279,  24346,  24413,  24480,  24547,  24613,  24680,  24746,
        24811,  24877,  24942,  25007,  25072,  25137,  25201,  25265,
        25329,  25393,  25456,  25519,  25582,  25645,  25708,  25770,
        25832,  25893,  25955,  26016,  26077,  26138,  26198,  26259,
        26319,  26378,  26438,  26497,  26556,  26615,  26674,  26732,
        26790,  26848,  26905,  26962,  27019,  27076,  27133,  27189,
        27245,  27300,  27356,  27411,  27466,  27521,  27575,  27629,
        27683,  27737,  27790,  27843,  27896,  27949,  28001,  28053,
        28105,  28157,  28208,  28259,  28310,  28360,  28411,  28460,
        28510,  28560,  28609,  28658,  28706,  28755,  28803,  28850,
        28898,  28945,  28992,  29039,  29085,  29131,  29177,  29223,
        29268,  29313,  29358,  29403,  29447,  29491,  29534,  29578,
        29621,  29664,  29706,  29749,  29791,  29832,  29874,  29915,
        29956,  29997,  30037,  30077,  30117,  30156,  30195,  30234,
        30273,  30311,  30349,  30387,  30424,  30462,  30498,  30535,
        30571,  30607,  30643,  30679,  30714,  30749,  30783,  30818,
        30852,  30885,  30919,  30952,  30985,  31017,  31050,  31082,
        31113,  31145,  31176,  31206,  31237,  31267,  31297,  31327,
        31356,  31385,  31414,  31442,  31470,  31498,  31526,  31553,
        31580,  31607,  31633,  31659,  31685,  31710,  31736,  31760,
        31785,  31809,  31833,  31857,  31880,  31903,  31926,  31949,
        31971,  31993,  32014,  32036,  32057,  32077,  32098,  32118,
        32137,  32157,  32176,  32195,  32213,  32232,  32250,  32267,
        32285,  32302,  32318,  32335,  32351,  32367,  32382,  32397,
        32412,  32427,  32441,  32455,  32469,  32482,  32495,  32508,
        32521,  32533,  32545,  32556,  32567,  32578,  32589,  32599,
        32609,  32619,  32628,  32637,  32646,  32655,  32663,  32671,
        32678,  32685,  32692,  32699,  32705,  32711,  32717,  32722,
        32728,  32732,  32737,  32741,  32745,  32748,  32752,  32755,
        32757,  32759,  32761,  32763,  32765,  32766,  32766,  32767,
        32767,  32767,  32766,  32766,  32765,  32763,  32761,  32759,
        32757,  32755,  32752,  32748,  32745,  32741,  32737,  32732,
        32728,  32722,  32717,  32711,  32705,  32699,  32692,  32685,
        32678,  32671,  32663,  32655,  32646,  32637,  32628,  32619,
        32609,  32599,  32589,  32578,  32567,  32556,  32545,  32533,
        32521,  32508,  32495,  32482,  32469,  32455,  32441,  32427,
        32412,  32397,  32382,  32367,  32351,  32335,  32318,  32302,
        32285,  32267,  32250,  32232,  32213,  32195,  32176,  32157,
        32137,  32118,  32098,  32077,  32057,  32036,  32014,  31993,
        31971,  31949,  31926,  31903,  31880,  31857,  31833,  31809,
        31785,  31760,  31736,  31710,  31685,  31659,  31633,  31607,
        31580,  31553,  31526,  31498,  31470,  31442,  31414,  31385,
        31356,  31327,  31297,  31267,  31237,  31206,  31176,  31145,
        31113,  31082,  31050,  31017,  30985,  30952,  30919,  30885,
        30852,  30818,  30783,  30749,  30714,  30679,  30643,  30607,
        30571,  30535,  30498,  30462,  30424,  30387,  30349,  30311,
        30273,  30234,  30195,  30156,  30117,  30077,  30037,  29997,
        29956,  29915,  29874,  29832,  29791,  29749,  29706,  29664,
        29621,  29578,  29534,  29491,  29447,  29403,  29358,  29313,
        29268,  29223,  29177,  29131,  29085,  29039,  28992,  28945,
        28898,  28850,  28803,  28755,  28706,  28658,  28609,  28560,
        28510,  28460,  28411,  28360,  28310,  28259,  28208,  28157,
        28105,  28053,  28001,  27949,  27896,  27843,  27790,  27737,
        27683,  27629,  27575,  27521,  27466,  27411,  27356,  27300,
        27245,  27189,  27133,  27076,  27019,  26962,  26905,  26848,
        26790,  26732,  26674,  26615,  26556,  26497,  26438,  26378,
        26319,  26259,  26198,  26138,  26077,  26016,  25955,  25893,
        25832,  25770,  25708,  25645,  25582,  25519,  25456,  25393,
        25329,  25265,  25201,  25137,  25072,  25007,  24942,  24877,
        24811,  24746,  24680,  24613,  24547,  24480,  24413,  24346,
        24279,  24211,  24143,  24075,  24007,  23938,  23870,  23801,
        23731,  23662,  23592,  23522,  23452,  23382,  23311,  23241,
        23170,  23099,  23027,  22956,  22884,  22812,  22739,  22667,
        22594,  22521,  22448,  22375,  22301,  22227,  22154,  22079,
        22005,  21930,  21856,  21781,  21705,  21630,  21554,  21479,
        21403,  21326,  21250,  21173,  21096,  21019,  20942,  20865,
        20787,  20709,  20631,  20553,  20475,  20396,  20317,  20238,
        20159,  20080,  20000,  19921,  19841,  19761,  19680,  19600,
        19519,  19438,  19357,  19276,  19195,  19113,  19032,  18950,
        18868,  18785,  18703,  18620,  18537,  18454,  18371,  18288,
        18204,  18121,  18037,  17953,  17869,  17784,  17700,  17615,
        17530,  17445,  17360,  17275,  17189,  17104,  17018,  16932,
        16846,  16759,  16673,  16586,  16499,  16413,  16325,  16238,
        16151,  16063,  15976,  15888,  15800,  15712,  15623,  15535,
        15446,  15358,  15269,  15180,  15090,  15001,  14912,  14822,
        14732,  14643,  14553,  14462,  14372,  14282,  14191,  14101,
        14010,  13919,  13828,  13736,  13645,  13554,  13462,  13370,
        13279,  13187,  13094,  13002,  12910,  12817,  12725,  12632,
        12539,  12446,  12353,  12260,  12167,  12074,  11980,  11886,
        11793,  11699,  11605,  11511,  11417,  11322,  11228,  11133,
        11039,  10944,  10849,  10754,  10659,  10564,  10469,  10374,
        10278,  10183,  10087,  9992,   9896,   9800,   9704,   9608,
        9512,   9416,   9319,   9223,   9126,   9030,   8933,   8836,
        8739,   8642,   8545,   8448,   8351,   8254,   8157,   8059,
        7962,   7864,   7767,   7669,   7571,   7473,   7375,   7277,
        7179,   7081,   6983,   6885,   6786,   6688,   6590,   6491,
        6393,   6294,   6195,   6096,   5998,   5899,   5800,   5701,
        5602,   5503,   5404,   5305,   5205,   5106,   5007,   4907,
        4808,   4708,   4609,   4509,   4410,   4310,   4210,   4111,
        4011,   3911,   3811,   3712,   3612,   3512,   3412,   3312,
        3212,   3112,   3012,   2911,   2811,   2711,   2611,   2511,
        2410,   2310,   2210,   2110,   2009,   1909,   1809,   1708,
        1608,   1507,   1407,   1307,   1206,   1106,   1005,   905,
        804,    704,    603,    503,    402,    302,    201,    101,
        0,  -101,   -201,   -302,   -402,   -503,   -603,   -704,
        -804,   -905,   -1005,  -1106,  -1206,  -1307,  -1407,  -1507,
        -1608,  -1708,  -1809,  -1909,  -2009,  -2110,  -2210,  -2310,
        -2410,  -2511,  -2611,  -2711,  -2811,  -2911,  -3012,  -3112,
        -3212,  -3312,  -3412,  -3512,  -3612,  -3712,  -3811,  -3911,
        -4011,  -4111,  -4210,  -4310,  -4410,  -4509,  -4609,  -4708,
        -4808,  -4907,  -5007,  -5106,  -5205,  -5305,  -5404,  -5503,
        -5602,  -5701,  -5800,  -5899,  -5998,  -6096,  -6195,  -6294,
        -6393,  -6491,  -6590,  -6688,  -6786,  -6885,  -6983,  -7081,
        -7179,  -7277,  -7375,  -7473,  -7571,  -7669,  -7767,  -7864,
        -7962,  -8059,  -8157,  -8254,  -8351,  -8448,  -8545,  -8642,
        -8739,  -8836,  -8933,  -9030,  -9126,  -9223,  -9319,  -9416,
        -9512,  -9608,  -9704,  -9800,  -9896,  -9992,  -10087, -10183,
        -10278, -10374, -10469, -10564, -10659, -10754, -10849, -10944,
        -11039, -11133, -11228, -11322, -11417, -11511, -11605, -11699,
        -11793, -11886, -11980, -12074, -12167, -12260, -12353, -12446,
        -12539, -12632, -12725, -12817, -12910, -13002, -13094, -13187,
        -13279, -13370, -13462, -13554, -13645, -13736, -13828, -13919,
        -14010, -14101, -14191, -14282, -14372, -14462, -14553, -14643,
        -14732, -14822, -14912, -15001, -15090, -15180, -15269, -15358,
        -15446, -15535, -15623, -15712, -15800, -15888, -15976, -16063,
        -16151, -16238, -16325, -16413, -16499, -16586, -16673, -16759,
        -16846, -16932, -17018, -17104, -17189, -17275, -17360, -17445,
        -17530, -17615, -17700, -17784, -17869, -17953, -18037, -18121,
        -18204, -18288, -18371, -18454, -18537, -18620, -18703, -18785,
        -18868, -18950, -19032, -19113, -19195, -19276, -19357, -19438,
        -19519, -19600, -19680, -19761, -19841, -19921, -20000, -20080,
        -20159, -20238, -20317, -20396, -20475, -20553, -20631, -20709,
        -20787, -20865, -20942, -21019, -21096, -21173, -21250, -21326,
        -21403, -21479, -21554, -21630, -21705, -21781, -21856, -21930,
        -22005, -22079, -22154, -22227, -22301, -22375, -22448, -22521,
        -22594, -22667, -22739, -22812, -22884, -22956, -23027, -23099,
        -23170, -23241, -23311, -23382, -23452, -23522, -23592, -23662,
        -23731, -23801, -23870, -23938, -24007, -24075, -24143, -24211,
        -24279, -24346, -24413, -24480, -24547, -24613, -24680, -24746,
        -24811, -24877, -24942, -25007, -25072, -25137, -25201, -25265,
        -25329, -25393, -25456, -25519, -25582, -25645, -25708, -25770,
        -25832, -25893, -25955, -26016, -26077, -26138, -26198, -26259,
        -26319, -26378, -26438, -26497, -26556, -26615, -26674, -26732,
        -26790, -26848, -26905, -26962, -27019, -27076, -27133, -27189,
        -27245, -27300, -27356, -27411, -27466, -27521, -27575, -27629,
        -27683, -27737, -27790, -27843, -27896, -27949, -28001, -28053,
        -28105, -28157, -28208, -28259, -28310, -28360, -28411, -28460,
        -28510, -28560, -28609, -28658, -28706, -28755, -28803, -28850,
        -28898, -28945, -28992, -29039, -29085, -29131, -29177, -29223,
        -29268, -29313, -29358, -29403, -29447, -29491, -29534, -29578,
        -29621, -29664, -29706, -29749, -29791, -29832, -29874, -29915,
        -29956, -29997, -30037, -30077, -30117, -30156, -30195, -30234,
        -30273, -30311, -30349, -30387, -30424, -30462, -30498, -30535,
        -30571, -30607, -30643, -30679, -30714, -30749, -30783, -30818,
        -30852, -30885, -30919, -30952, -30985, -31017, -31050, -31082,
        -31113, -31145, -31176, -31206, -31237, -31267, -31297, -31327,
        -31356, -31385, -31414, -31442, -31470, -31498, -31526, -31553,
        -31580, -31607, -31633, -31659, -31685, -31710, -31736, -31760,
        -31785, -31809, -31833, -31857, -31880, -31903, -31926, -31949,
        -31971, -31993, -32014, -32036, -32057, -32077, -32098, -32118,
        -32137, -32157, -32176, -32195, -32213, -32232, -32250, -32267,
        -32285, -32302, -32318, -32335, -32351, -32367, -32382, -32397,
        -32412, -32427, -32441, -32455, -32469, -32482, -32495, -32508,
        -32521, -32533, -32545, -32556, -32567, -32578, -32589, -32599,
        -32609, -32619, -32628, -32637, -32646, -32655, -32663, -32671,
        -32678, -32685, -32692, -32699, -32705, -32711, -32717, -32722,
        -32728, -32732, -32737, -32741, -32745, -32748, -32752, -32755,
        -32757, -32759, -32761, -32763, -32765, -32766, -32766, -32767,
        -32767, -32767, -32766, -32766, -32765, -32763, -32761, -32759,
        -32757, -32755, -32752, -32748, -32745, -32741, -32737, -32732,
        -32728, -32722, -32717, -32711, -32705, -32699, -32692, -32685,
        -32678, -32671, -32663, -32655, -32646, -32637, -32628, -32619,
        -32609, -32599, -32589, -32578, -32567, -32556, -32545, -32533,
        -32521, -32508, -32495, -32482, -32469, -32455, -32441, -32427,
        -32412, -32397, -32382, -32367, -32351, -32335, -32318, -32302,
        -32285, -32267, -32250, -32232, -32213, -32195, -32176, -32157,
        -32137, -32118, -32098, -32077, -32057, -32036, -32014, -31993,
        -31971, -31949, -31926, -31903, -31880, -31857, -31833, -31809,
        -31785, -31760, -31736, -31710, -31685, -31659, -31633, -31607,
        -31580, -31553, -31526, -31498, -31470, -31442, -31414, -31385,
        -31356, -31327, -31297, -31267, -31237, -31206, -31176, -31145,
        -31113, -31082, -31050, -31017, -30985, -30952, -30919, -30885,
        -30852, -30818, -30783, -30749, -30714, -30679, -30643, -30607,
        -30571, -30535, -30498, -30462, -30424, -30387, -30349, -30311,
        -30273, -30234, -30195, -30156, -30117, -30077, -30037, -29997,
        -29956, -29915, -29874, -29832, -29791, -29749, -29706, -29664,
        -29621, -29578, -29534, -29491, -29447, -29403, -29358, -29313,
        -29268, -29223, -29177, -29131, -29085, -29039, -28992, -28945,
        -28898, -28850, -28803, -28755, -28706, -28658, -28609, -28560,
        -28510, -28460, -28411, -28360, -28310, -28259, -28208, -28157,
        -28105, -28053, -28001, -27949, -27896, -27843, -27790, -27737,
        -27683, -27629, -27575, -27521, -27466, -27411, -27356, -27300,
        -27245, -27189, -27133, -27076, -27019, -26962, -26905, -26848,
        -26790, -26732, -26674, -26615, -26556, -26497, -26438, -26378,
        -26319, -26259, -26198, -26138, -26077, -26016, -25955, -25893,
        -25832, -25770, -25708, -25645, -25582, -25519, -25456, -25393,
        -25329, -25265, -25201, -25137, -25072, -25007, -24942, -24877,
        -24811, -24746, -24680, -24613, -24547, -24480, -24413, -24346,
        -24279, -24211, -24143, -24075, -24007, -23938, -23870, -23801,
        -23731, -23662, -23592, -23522, -23452, -23382, -23311, -23241,
        -23170, -23099, -23027, -22956, -22884, -22812, -22739, -22667,
        -22594, -22521, -22448, -22375, -22301, -22227, -22154, -22079,
        -22005, -21930, -21856, -21781, -21705, -21630, -21554, -21479,
        -21403, -21326, -21250, -21173, -21096, -21019, -20942, -20865,
        -20787, -20709, -20631, -20553, -20475, -20396, -20317, -20238,
        -20159, -20080, -20000, -19921, -19841, -19761, -19680, -19600,
        -19519, -19438, -19357, -19276, -19195, -19113, -19032, -18950,
        -18868, -18785, -18703, -18620, -18537, -18454, -18371, -18288,
        -18204, -18121, -18037, -17953, -17869, -17784, -17700, -17615,
        -17530, -17445, -17360, -17275, -17189, -17104, -17018, -16932,
        -16846, -16759, -16673, -16586, -16499, -16413, -16325, -16238,
        -16151, -16063, -15976, -15888, -15800, -15712, -15623, -15535,
        -15446, -15358, -15269, -15180, -15090, -15001, -14912, -14822,
        -14732, -14643, -14553, -14462, -14372, -14282, -14191, -14101,
        -14010, -13919, -13828, -13736, -13645, -13554, -13462, -13370,
        -13279, -13187, -13094, -13002, -12910, -12817, -12725, -12632,
        -12539, -12446, -12353, -12260, -12167, -12074, -11980, -11886,
        -11793, -11699, -11605, -11511, -11417, -11322, -11228, -11133,
        -11039, -10944, -10849, -10754, -10659, -10564, -10469, -10374,
        -10278, -10183, -10087, -9992,  -9896,  -9800,  -9704,  -9608,
        -9512,  -9416,  -9319,  -9223,  -9126,  -9030,  -8933,  -8836,
        -8739,  -8642,  -8545,  -8448,  -8351,  -8254,  -8157,  -8059,
        -7962,  -7864,  -7767,  -7669,  -7571,  -7473,  -7375,  -7277,
        -7179,  -7081,  -6983,  -6885,  -6786,  -6688,  -6590,  -6491,
        -6393,  -6294,  -6195,  -6096,  -5998,  -5899,  -5800,  -5701,
        -5602,  -5503,  -5404,  -5305,  -5205,  -5106,  -5007,  -4907,
        -4808,  -4708,  -4609,  -4509,  -4410,  -4310,  -4210,  -4111,
        -4011,  -3911,  -3811,  -3712,  -3612,  -3512,  -3412,  -3312,
        -3212,  -3112,  -3012,  -2911,  -2811,  -2711,  -2611,  -2511,
        -2410,  -2310,  -2210,  -2110,  -2009,  -1909,  -1809,  -1708,
        -1608,  -1507,  -1407,  -1307,  -1206,  -1106,  -1005,  -905,
        -804,   -704,   -603,   -503,   -402,   -302,   -201,   -101
        };

        #endregion

        #region "==================== ENUMS ===================================================
        public enum detune_modes
        {
            dm_cents = 0,
            dm_hz = 1
        };

        public enum morph_modes
        {
            linear = 0,
            exponential = 1
        }
        public enum used_midi_cc
        {
            ucc_none = 0,
            ucc_pitch_bend = 1,
            ucc_modulation = 2,
            ucc_breath_controller = 3,
            ucc_expression = 4,
            ucc_foot_controller = 5,
            ucc_sustain = 6,
            ucc_velocity = 7
        };

        public enum envelope
        {
            env_amplitude = 0,
            env_noise = 1,
            env_noise_cutoff_level = 2,
            env_timbre = 3,
            env_sample = 4,
            env_pitch = 5,
            env_portamento = 6,
            env_tremolo = 7,
            env_vibrato = 8,
            env_timbre_lfo = 9
        };

        public enum env_types
        {
            env_type_none = 0,
            env_type_linear = 1,
            env_type_exponential = 2
        };

        public enum lfos
        {
            lfo_tremolo = 0,
            lfo_vibrato = 1,
            lfo_timbre = 2
        };

        public enum effect
        {
            eff_none = 0,
            eff_soft_clipping = 1
        }

        public enum patch_types
        {
            pt_general_midi = 0,
            pt_user = 1
        }

        public enum inharmonic_samples
        {
            inharmonic_sample_none = 0,
            inharmonic_sample_hammer1 = 1,
            inharmonic_sample_hammer2 = 2,
            inharmonic_sample_hammer3 = 3,
            inharmonic_sample_hammer4 = 4,
            inharmonic_sample_bowscrape1 = 5,
            inharmonic_sample_bowscrape2 = 6,
            inharmonic_sample_keyup = 7,
            inharmonic_sample_keyclick = 8,
            inharmonic_sample_KS_very_long_not_filtered = 9,
            inharmonic_sample_KS_long_not_filtered = 10,
            inharmonic_sample_KS_normal_not_filtered = 11,
            inharmonic_sample_KS_short_not_filtered = 12,
            inharmonic_sample_KS_very_short_not_filtered = 13,
            inharmonic_sample_KS_very_long_filtered = 14,
            inharmonic_sample_KS_long_filtered = 15,
            inharmonic_sample_KS_normal_filtered = 16,
            inharmonic_sample_KS_short_filtered = 17,
            inharmonic_sample_KS_very_short_filtered = 18

        };

        public enum inharmonic_sample_playback_modes
        {
            sample_playback_mode_one_shot = 1,
            sample_playback_mode_looped = 2
        };

        public enum timbre_modes
        {
            timbre_mode_2d = 2,
            timbre_envelope = 3
        };

        public enum phase_settings
        {
            phase_all_in_phase = 0,
            phase_inv_even = 1,
            phase_inv_odd = 2,
            phase_inv_3711 = 3
        };

        public enum adsr_sections
        {
            adsr_section_attack = 0,
            adsr_section_decay = 1,
            adsr_section_sustain = 2,
            adsr_section_release = 3
        };

        public enum filter_types
        {
            lowpass = 0,
            highpass = 1,
            bandpass = 3,
            notch = 4,
            peak = 5,
            lowshelf = 6,
            highshelf = 7
        };

        public enum msg_types
        {
            msg_type_adsr_section = 0,
            msg_type_adsr_section_envelope_config = 1,
            msg_type_biquad_eq_calculated = 2,
            msg_type_lfo_envelope_config = 3,
            msg_type_patch = 4,
            msg_type_waveform = 5,
            msg_type_waveform_harmonic = 6,
            msg_type_waveform_set = 7,
            msg_type_patch_all = 8,
            SaveConfig = 9,
            msg_type_body_resonance_filter_band = 10,
            msg_type_envelope_control = 11,
            msg_type_save_patch_to_EEPROM = 15,
            msg_type_load_patch_from_EEPROM = 16,
            msg_type_performance = 17,
            msg_type_select_performance = 18,
            msg_type_relay_performances_to_mixer = 19,
            InstanceNoteOn = 20,
            InstanceNoteOff = 21,
            InstanceControlChange = 22,
            ChangeEQ = 30,
            ChangeEffect = 31,
            ChannelVolume = 40,
            StartPlayMode = 41,
            StopPlayMode = 42,
            msg_default_banks = 43,
            StartPlayModeTP = 36860,
            StopPlayModeTP = 36861,
            StartStopPlayModeTP_Replacement = 36862
        };

        public enum msg_recipients
        {
            mixer = 9,
            master = 10,
            no_recipient = 100,
        };

        public enum lfo_wave_types
        {
            sine = 0,
            square = 1,
            triangle = 2,
            saw = 3
        };

        public enum ks_types
        {
            section = 0,
            envelope = 1
        };
        public enum ks_items
        {
            section_duration = 0,
            envelope_target = 1,
            envelope_linear_delta = 2,
            envelope_exp_mult_delta = 3
        };

        public enum adsr_presets
        {
            env_pres_amplitude_HammeredString,
            env_pres_amplitude_SlowAttack,
            env_pres_amplitude_VerySlowAttack,
            env_pres_amplitude_Organlike,
            env_pres_tremolo_DeepAndSlow,
            env_pres_tremolo_ShallowAndSlow,
            env_pres_tremolo_DeepAndFast,
            env_pres_tremolo_ShallowAndFast,
            env_pres_tremolo_BulkSetLinearDSR,
            env_pres_tremolo_BulkSetExpDSR,
            env_pres_tremolo_Clear,
            env_pres_vibrato_DeepAndSlow,
            env_pres_vibrato_ShallowAndSlow,
            env_pres_vibrato_DeepAndFast,
            env_pres_vibrato_ShallowAndFast,
            env_pres_vibrato_BulkSetLinearADSR,
            env_pres_vibrato_BulkSetExpADSR,
            env_pres_vibrato_Clear,
            env_pres_timbre_FastSweepUp,
            env_pres_timbre_SlowSweepUp,
            env_pres_timbre_ExpVeryFastSweepUp,
            env_pres_timbre_FastSweepDown,
            env_pres_timbre_SlowSweepDown,
            env_pres_timbre_ExpVeryFastSweepDown,
            env_pres_timbre_ExpFastSweepUp,
            env_pres_timbre_ExpSlowSweepUp,
            env_pres_timbre_ExpFastSweepDown,
            env_pres_timbre_ExpSlowSweepDown,
            env_pres_timbre_BulkSetLinearADSR,
            env_pres_timbre_BulkSetExpADSR,
            env_pres_timbre_Clear,
            env_pres_noise_QuickSoftPuff,
            env_pres_noise_SlowSoftPuff,
            env_pres_noise_QuickHardPuff,
            env_pres_noise_SlowHardPuff,
            env_pres_noise_Clear,
            env_pres_noise_co_apply_lpf_exp_fast,
            env_pres_noise_co_apply_lpf_exp_slow,
            env_pres_noise_co_BulkSetLinearADSR,
            env_pres_noise_co_BulkSetExpADSR,
            env_pres_noise_co_Clear,
            env_pres_pitch_fast_linear_drop,
            env_pres_pitch_slow_linear_drop,
            env_pres_pitch_fast_linear_rise,
            env_pres_pitch_slow_linear_rise,
            env_pres_pitch_fast_exp_drop,
            env_pres_pitch_slow_exp_drop,
            env_pres_pitch_fast_exp_rise,
            env_pres_pitch_slow_exp_rise,
            env_pres_pitch_BulkSetLinearADSR,
            env_pres_pitch_BulkSetExpADSR,
            env_pres_pitch_Clear,
            env_pres_sample_AttackOnLevel100,
            env_pres_sample_AttackOnLevel75,
            env_pres_sample_AttackOnLevel50,
            env_pres_sample_AttackOnLevel25,
            env_pres_sample_SustainOnLevel25,
            env_pres_sample_ReleaseOnLevel25,
            env_pres_sample_AttackOnLevel10,
            env_pres_sample_SustainOnLevel10,
            env_pres_sample_ReleaseOnLevel10,
            env_pres_sample_AttackOnLevel5,
            env_pres_sample_SustainOnLevel5,
            env_pres_sample_ReleaseOnLevel5,
            env_pres_sample_KS_100,
            env_pres_sample_KS_75,
            env_pres_sample_KS_50,
            env_pres_sample_KS_25,
            env_pres_sample_Clear,
            env_pres_timbre_lfo_DeepAndSlow,
            env_pres_timbre_lfo_ShallowAndSlow,
            env_pres_timbre_lfo_DeepAndFast,
            env_pres_timbre_lfo_ShallowAndFast,
            env_pres_timbre_lfo_BulkSetLinearDSR,
            env_pres_timbre_lfo_BulkSetExpDSR,
            env_pres_timbre_lfo_Clear
        };

        public enum directions
        {
            up = 0,
            down = 1
        }

        public enum dbchanges
        {
            nochange,
            minus_3db,
            minus_6db,
            minus_12db,
            minus_24db,
            minus_48db,
            plus_3db,
            plus_6db,
            plus_12db,
            plus_24db,
            plus_48db
        }

        public enum harmonic_level_shape_types
        {
            all_harmonics_zero,
            all_harmonics_max,
            linearly_descreasing,
            exponential_all,
            exponential_odd
        }

        public enum harmonic_phase_shape_types
        {
            invert_even,
            invert_odd,
            invert_3711,
            all_in_phase
        }

        public enum instrument_types
        {
            violin,
            cello,
            clarinet,
            trumpet,
            flute,
            guitar,
            muted_guitar
        }

        public enum waveform_shapes
        {
            square,
            triangle,
            saw
        }

        public enum phases
        {
            Normal,
            Inverted
        }

        public enum note_sector_range_contexts
        {
            MorphWholeSector,
            MorphCurrentIntensityLayerBlocks,
            MorphCurrentIntensityLayerWaveforms,
            CopyWholeSector,
            CopyCurrentBlock,
            CopyCurrentWaveform
        }

        public enum intensity_layer_range_contexts
        {
            MorphWholeLayer,
            MorphCurrentNoteSectorBlocks,
            MorphCurrentNoteSectorWaveforms,
            CopyWholeLayer,
            CopyCurrentBlock,
            CopyCurrentWaveform
        }

        public enum visual3d_modes
        {
            waveforms,
            harmonics
        }

        public enum visual3d_across_modes
        {
            waveform_block,
            note_sector,
            intensity_layer
        }

        #endregion

        #region "==================== PROPERTIES ========================"
        #endregion

        #region "==================== METHODS ==========================="
        public DA_Spectral()
        {
            data_changed = new c_data_changed_overview();
        }

        public static double get_dbchange_value(DA_Spectral.dbchanges c)
        {
            switch (c)
            {
                case dbchanges.minus_12db:
                    return -12;
                case dbchanges.minus_24db:
                    return -24;
                case dbchanges.minus_3db:
                    return -3;
                case dbchanges.minus_48db:
                    return -48;
                case dbchanges.minus_6db:
                    return -6;
                case dbchanges.nochange:
                    return 0;
                case dbchanges.plus_12db:
                    return 12;
                case dbchanges.plus_24db:
                    return 24;
                case dbchanges.plus_3db:
                    return 3;
                case dbchanges.plus_48db:
                    return 48;
                case dbchanges.plus_6db:
                    return 6;
                default:
                    return 0;
            }
        }

        public static string GetSectionNameFromID(adsr_sections section_id)
        {
            switch (section_id)
                {
                case adsr_sections.adsr_section_attack:
                    return "Attack";
                case adsr_sections.adsr_section_decay:
                    return "Decay";
                case adsr_sections.adsr_section_sustain:
                    return "Sustain";
                case adsr_sections.adsr_section_release:
                    return "Release";
                default:
                    return "";
            }
        }

        public static Int32 GetSectionIDFromName(string section_id)
        {
            switch (section_id.ToUpper())
            {
                case "ATTACK":
                    return (int)adsr_sections.adsr_section_attack;
                case "DECAY":
                    return (int)adsr_sections.adsr_section_decay;
                case "SUSTAIN":
                    return (int)adsr_sections.adsr_section_sustain;
                case "RELEASE":
                    return (int)adsr_sections.adsr_section_release;
                default:
                    return 0;
            }
        }

        public static string GetCCNameFromID(used_midi_cc cc)
        {
            switch (cc)
            {
                case used_midi_cc.ucc_breath_controller:
                    return "BreathController";
                case used_midi_cc.ucc_expression:
                    return "Expression";
                case used_midi_cc.ucc_foot_controller:
                    return "FootController";
                case used_midi_cc.ucc_modulation:
                    return "Modulation";
                case used_midi_cc.ucc_none:
                    return "None";
                case used_midi_cc.ucc_pitch_bend:
                    return "PitchBend";
                case used_midi_cc.ucc_sustain:
                    return "Sustain";
                case used_midi_cc.ucc_velocity:
                    return "Velocity";
                default:
                    return "";
            }
        }

        public static Int32 GetCCIDFromName(string cc)
        {
            switch (cc.ToUpper())
            {
                case "BREATHCONTROLLER":
                    return (int)used_midi_cc.ucc_breath_controller;
                case "EXPRESSION":
                    return (int)used_midi_cc.ucc_expression;
                case "FOOTCONTROLLER":
                    return (int)used_midi_cc.ucc_foot_controller;
                case "MODULATION":
                    return (int)used_midi_cc.ucc_modulation;
                case "NONE":
                    return (int)used_midi_cc.ucc_none;
                case "PITCHBEND":
                    return (int)used_midi_cc.ucc_pitch_bend;
                case "SUSTAIN":
                    return (int)used_midi_cc.ucc_sustain;
                case "VELOCITY":
                    return (int)used_midi_cc.ucc_velocity;
                default:
                    return 0;
            }
        }
        public static string GetWaveShapeNameFromID(waveform_shapes s)
        {
            switch (s)
            {
                case waveform_shapes.saw:
                    return "Saw";
                case waveform_shapes.square:
                    return "Square";
                case waveform_shapes.triangle:
                    return "Triangle";
                default:
                    return "";
            }
        }

        public static string GetWaveShapeNameFromID(lfo_wave_types t)
        {
            switch (t)
            {
                case lfo_wave_types.saw:
                    return "Saw";
                case lfo_wave_types.square:
                    return "Square";
                case lfo_wave_types.triangle:
                    return "Triangle";
                case lfo_wave_types.sine:
                    return "Sine";
                default:
                    return "";
            }
        }

        public static Int32 GetWaveShapeIDFromName(string t)
        {
            switch (t.ToUpper())
            {
                case "SAW":
                    return (int)lfo_wave_types.saw;
                case "SQUARE":
                    return (int)lfo_wave_types.square;
                case "TRIANGLE":
                    return (int)lfo_wave_types.triangle;
                case "SINE":
                    return (int)lfo_wave_types.sine;
                default:
                    return 0;
            }
        }

        public static string GetLFONameFromID(lfos l)
        {
            switch (l)
            {
                case lfos.lfo_timbre:
                    return "TimbreLFO";
                case lfos.lfo_tremolo:
                    return "Tremolo";
                case lfos.lfo_vibrato:
                    return "Vibrato";
                default:
                    return "";
            }
        }

        public static int GetLFOIDFromName(string l)
        {
            switch (l.ToUpper())
            {
                case "TIMBRELFO":
                    return (int)lfos.lfo_timbre;
                case "TREMOLO":
                    return (int)lfos.lfo_tremolo;
                case "VIBRATO":
                    return (int)lfos.lfo_vibrato;
                default:
                    return 0;
            }
        }

        public static string GetEnvelopeNameFromID(envelope e)
        {
            switch (e)
            {
                case envelope.env_amplitude:
                    return "Amplitude";
                case envelope.env_noise:
                    return "NoiseGain";
                case envelope.env_noise_cutoff_level:
                    return "NoiseCutoffFrequency";
                case envelope.env_pitch:
                    return "PitchShift";
                case envelope.env_portamento:
                    return "Portamento";
                case envelope.env_sample:
                    return "SampleGain";
                case envelope.env_timbre:
                    return "TimbreMorph";
                case envelope.env_timbre_lfo:
                    return "TimbreLFODepth";
                case envelope.env_tremolo:
                    return "TremoloDepth";
                case envelope.env_vibrato:
                    return "VibratoDepth";
                default:
                    return "";
            }
        }

        public static Int32 GetEnvelopeIDFromName(string e)
        {
            switch (e.ToUpper())
            {
                case "AMPLITUDE":
                    return (int)envelope.env_amplitude;
                case "NOISEGAIN":
                    return (int)envelope.env_noise;
                case "NOISECUTOFFFREQUENCY":
                    return (int)envelope.env_noise_cutoff_level;
                case "PITCHSHIFT":
                    return (int)envelope.env_pitch;
                case "PORTAMENTO":
                    return (int)envelope.env_portamento;
                case "SAMPLEGAIN":
                    return (int)envelope.env_sample;
                case "TIMBREMORPH":
                    return (int)envelope.env_timbre;
                case "TIMBRELFODEPTH":
                    return (int)envelope.env_timbre_lfo;
                case "TREMOLODEPTH":
                    return (int)envelope.env_tremolo;
                case "VIBRATODEPTH":
                    return (int)envelope.env_vibrato;
                default:
                    return 0;
            }
        }

        public static string GetSampleNameFromID(inharmonic_samples s)
        {
            switch (s)
            {
                case inharmonic_samples.inharmonic_sample_bowscrape1:
                    return "BowScrape1";
                case inharmonic_samples.inharmonic_sample_bowscrape2:
                    return "BowScrape2";
                case inharmonic_samples.inharmonic_sample_hammer1:
                    return "KeyHammer1";
                case inharmonic_samples.inharmonic_sample_hammer2:
                    return "KeyHammer2";
                case inharmonic_samples.inharmonic_sample_hammer3:
                    return "KeyHammer3";
                case inharmonic_samples.inharmonic_sample_hammer4:
                    return "KeyHammer4";
                case inharmonic_samples.inharmonic_sample_keyclick:
                    return "KeyClick";
                case inharmonic_samples.inharmonic_sample_keyup:
                    return "KeyUp";
                case inharmonic_samples.inharmonic_sample_KS_long_filtered:
                    return "KSLongF";
                case inharmonic_samples.inharmonic_sample_KS_long_not_filtered:
                    return "KSLong";
                case inharmonic_samples.inharmonic_sample_KS_normal_filtered:
                    return "KSStdF";
                case inharmonic_samples.inharmonic_sample_KS_normal_not_filtered:
                    return "KSStd";
                case inharmonic_samples.inharmonic_sample_KS_short_filtered:
                    return "KSShortF";
                case inharmonic_samples.inharmonic_sample_KS_short_not_filtered:
                    return "KSShort";
                case inharmonic_samples.inharmonic_sample_KS_very_long_filtered:
                    return "KSVLongF";
                case inharmonic_samples.inharmonic_sample_KS_very_long_not_filtered:
                    return "KSVLong";
                case inharmonic_samples.inharmonic_sample_KS_very_short_filtered:
                    return "KSVShortF";
                case inharmonic_samples.inharmonic_sample_KS_very_short_not_filtered:
                    return "KSVShort";
                case inharmonic_samples.inharmonic_sample_none:
                    return "None";
                default:
                    return "";
            }
        }

        public static Int32 GetSampleIDFromName(string s)
        {
            switch (s.ToUpper())
            {
                case "BOWSCRAPE1":
                    return (int)inharmonic_samples.inharmonic_sample_bowscrape1;
                case "BOWSCRAPE2":
                    return (int)inharmonic_samples.inharmonic_sample_bowscrape2;
                case "KEYHAMMER1":
                    return (int)inharmonic_samples.inharmonic_sample_hammer1;
                case "KEYHAMMER2":
                    return (int)inharmonic_samples.inharmonic_sample_hammer2;
                case "KEYHAMMER3":
                    return (int)inharmonic_samples.inharmonic_sample_hammer3;
                case "KEYHAMMER4":
                    return (int)inharmonic_samples.inharmonic_sample_hammer4;
                case "KEYCLICK":
                    return (int)inharmonic_samples.inharmonic_sample_keyclick;
                case "KEYUP":
                    return (int)inharmonic_samples.inharmonic_sample_keyup;
                case "KSLONGF":
                    return (int)inharmonic_samples.inharmonic_sample_KS_long_filtered;
                case "KSLONG":
                    return (int)inharmonic_samples.inharmonic_sample_KS_long_not_filtered;
                case "KSSTDF":
                    return (int)inharmonic_samples.inharmonic_sample_KS_normal_filtered;
                case "KSSTD":
                    return (int)inharmonic_samples.inharmonic_sample_KS_normal_not_filtered;
                case "KSSHORTF":
                    return (int)inharmonic_samples.inharmonic_sample_KS_short_filtered;
                case "KSSHORT":
                    return (int)inharmonic_samples.inharmonic_sample_KS_short_not_filtered;
                case "KSVLONGF":
                    return (int)inharmonic_samples.inharmonic_sample_KS_very_long_filtered;
                case "KSVLONG":
                    return (int)inharmonic_samples.inharmonic_sample_KS_very_long_not_filtered;
                case "KSVSHORTF":
                    return (int)inharmonic_samples.inharmonic_sample_KS_very_short_filtered;
                case "KSVSHORT":
                    return (int)inharmonic_samples.inharmonic_sample_KS_very_short_not_filtered;
                case "NONE":
                    return (int)inharmonic_samples.inharmonic_sample_none;
                default:
                    return 0;
            }
        }

        public static string GetSampleModeFromID(inharmonic_sample_playback_modes s)
        {
            switch (s)
            {
                case inharmonic_sample_playback_modes.sample_playback_mode_looped:
                    return "Looped";
                case inharmonic_sample_playback_modes.sample_playback_mode_one_shot:
                    return "OneShot";
                default:
                    return "";
            }
        }

        public static Int32 GetSampleModeIDFromName(string s)
        {
            switch (s.ToUpper())
            {
                case "LOOPED":
                    return (int)inharmonic_sample_playback_modes.sample_playback_mode_looped;
                case "ONESHOT":
                    return (int)inharmonic_sample_playback_modes.sample_playback_mode_one_shot;
                default:
                    return 0;
            }
        }

        public static string GetEnvelopeTypeFromID(env_types s)
        {
            switch (s)
            {
                case env_types.env_type_exponential:
                    return "Exponential";
                case env_types.env_type_linear:
                    return "Linear";
                case env_types.env_type_none:
                    return "None";
                default:
                    return "";
            }
        }

        public static Int32 GetEnvelopeTypeIDFromName(string s)
        {
            switch (s.ToUpper())
            {
                case "EXPONENTIAL":
                    return (int)env_types.env_type_exponential;
                case "LINEAR":
                    return (int)env_types.env_type_linear;
                case "NONE":
                    return (int)env_types.env_type_none;
                default:
                    return 0;
            }
        }

        public static string GetDetuningModeFromID(detune_modes m)
        {
            switch (m)
            {
                case detune_modes.dm_cents:
                    return "Cents";
                case detune_modes.dm_hz:
                    return "Hz";
                default:
                    return "";
            }
        }

        public static Int32 GetDetuningModeIDFromName(string m)
        {
            switch (m.ToUpper())
            {
                case "CENTS":
                    return (int)detune_modes.dm_cents;
                case "HZ":
                    return (int)detune_modes.dm_hz;
                default:
                    return 0;
            }
        }

        public static string GetMIDINoteNameFromID(int note_id)
        {
            string note_name = "";

            int octave_number = (note_id / DA_Spectral.notes_per_octave) - 1;
            int octave_note_number = note_id % DA_Spectral.notes_per_octave;
            note_name = BL_MIDI.NoteNumberToName(octave_note_number) + octave_number.ToString();

            return note_name;
        }

        public static Int32 GetADSRPresetIDFromName(string p)
        {
            switch (p.ToUpper())
            {
                case "AMPLITUDE_HAMMEREDSTRING": return 0;
                case "AMPLITUDE_SLOWATTACK": return 1;
                case "AMPLITUDE_VERYSLOWATTACK": return 2;
                case "AMPLITUDE_ORGANLIKE": return 3;
                case "TREMOLO_DEEPANDSLOW": return 4;
                case "TREMOLO_SHALLOWANDSLOW": return 5;
                case "TREMOLO_DEEPANDFAST": return 6;
                case "TREMOLO_SHALLOWANDFAST": return 7;
                case "TREMOLO_BULKSETLINEARDSR": return 8;
                case "TREMOLO_BULKSETEXPDSR": return 9;
                case "TREMOLO_CLEAR": return 10;
                case "VIBRATO_DEEPANDSLOW": return 11;
                case "VIBRATO_SHALLOWANDSLOW": return 12;
                case "VIBRATO_DEEPANDFAST": return 13;
                case "VIBRATO_SHALLOWANDFAST": return 14;
                case "VIBRATO_BULKSETLINEARADSR": return 15;
                case "VIBRATO_BULKSETEXPADSR": return 16;
                case "VIBRATO_CLEAR": return 17;
                case "TIMBRE_FASTSWEEPUP": return 18;
                case "TIMBRE_SLOWSWEEPUP": return 19;
                case "TIMBRE_EXPVERYFASTSWEEPUP": return 20;
                case "TIMBRE_FASTSWEEPDOWN": return 21;
                case "TIMBRE_SLOWSWEEPDOWN": return 22;
                case "TIMBRE_EXPVERYFASTSWEEPDOWN": return 23;
                case "TIMBRE_EXPFASTSWEEPUP": return 24;
                case "TIMBRE_EXPSLOWSWEEPUP": return 25;
                case "TIMBRE_EXPFASTSWEEPDOWN": return 26;
                case "TIMBRE_EXPSLOWSWEEPDOWN": return 27;
                case "TIMBRE_BULKSETLINEARADSR": return 28;
                case "TIMBRE_BULKSETEXPADSR": return 29;
                case "TIMBRE_CLEAR": return 30;
                case "NOISE_QUICKSOFTPUFF": return 31;
                case "NOISE_SLOWSOFTPUFF": return 32;
                case "NOISE_QUICKHARDPUFF": return 33;
                case "NOISE_SLOWHARDPUFF": return 34;
                case "NOISE_CLEAR": return 35;
                case "NOISE_CO_APPLY_LPF_EXP_FAST": return 36;
                case "NOISE_CO_APPLY_LPF_EXP_SLOW": return 37;
                case "NOISE_CO_BULKSETLINEARADSR": return 38;
                case "NOISE_CO_BULKSETEXPADSR": return 39;
                case "NOISE_CO_CLEAR": return 40;
                case "PITCH_FAST_LINEAR_DROP": return 41;
                case "PITCH_SLOW_LINEAR_DROP": return 42;
                case "PITCH_FAST_LINEAR_RISE": return 43;
                case "PITCH_SLOW_LINEAR_RISE": return 44;
                case "PITCH_FAST_EXP_DROP": return 45;
                case "PITCH_SLOW_EXP_DROP": return 46;
                case "PITCH_FAST_EXP_RISE": return 47;
                case "PITCH_SLOW_EXP_RISE": return 48;
                case "PITCH_BULKSETLINEARADSR": return 49;
                case "PITCH_BULKSETEXPADSR": return 50;
                case "PITCH_CLEAR": return 51;
                case "SAMPLE_ATTACKONLEVEL100": return 52;
                case "SAMPLE_ATTACKONLEVEL75": return 53;
                case "SAMPLE_ATTACKONLEVEL50": return 54;
                case "SAMPLE_ATTACKONLEVEL25": return 55;
                case "SAMPLE_SUSTAINONLEVEL25": return 56;
                case "SAMPLE_RELEASEONLEVEL25": return 57;
                case "SAMPLE_ATTACKONLEVEL10": return 58;
                case "SAMPLE_SUSTAINONLEVEL10": return 59;
                case "SAMPLE_RELEASEONLEVEL10": return 60;
                case "SAMPLE_ATTACKONLEVEL5": return 61;
                case "SAMPLE_SUSTAINONLEVEL5": return 62;
                case "SAMPLE_RELEASEONLEVEL5": return 63;
                case "SAMPLE_KS_100": return 64;
                case "SAMPLE_KS_75": return 65;
                case "SAMPLE_KS_50": return 66;
                case "SAMPLE_KS_25": return 67;
                case "SAMPLE_CLEAR": return 68;
                case "TIMBRE_LFO_DEEPANDSLOW": return 69;
                case "TIMBRE_LFO_SHALLOWANDSLOW": return 70;
                case "TIMBRE_LFO_DEEPANDFAST": return 71;
                case "TIMBRE_LFO_SHALLOWANDFAST": return 72;
                case "TIMBRE_LFO_BULKSETLINEARDSR": return 73;
                case "TIMBRE_LFO_BULKSETEXPDSR": return 74;
                case "TIMBRE_LFO_CLEAR": return 75;

            }

            return 0;
        }

        public static string GetADSRPresetNameFromID(Int32 p)
        {
            switch (p)
            {
                case 0: return "amplitude_HammeredString";
                case 1: return "amplitude_SlowAttack";
                case 2: return "amplitude_VerySlowAttack";
                case 3: return "amplitude_Organlike";
                case 4: return "tremolo_DeepAndSlow";
                case 5: return "tremolo_ShallowAndSlow";
                case 6: return "tremolo_DeepAndFast";
                case 7: return "tremolo_ShallowAndFast";
                case 8: return "tremolo_BulkSetLinearDSR";
                case 9: return "tremolo_BulkSetExpDSR";
                case 10: return "tremolo_Clear";
                case 11: return "vibrato_DeepAndSlow";
                case 12: return "vibrato_ShallowAndSlow";
                case 13: return "vibrato_DeepAndFast";
                case 14: return "vibrato_ShallowAndFast";
                case 15: return "vibrato_BulkSetLinearADSR";
                case 16: return "vibrato_BulkSetExpADSR";
                case 17: return "vibrato_Clear";
                case 18: return "timbre_FastSweepUp";
                case 19: return "timbre_SlowSweepUp";
                case 20: return "timbre_ExpVeryFastSweepUp";
                case 21: return "timbre_FastSweepDown";
                case 22: return "timbre_SlowSweepDown";
                case 23: return "timbre_ExpVeryFastSweepDown";
                case 24: return "timbre_ExpFastSweepUp";
                case 25: return "timbre_ExpSlowSweepUp";
                case 26: return "timbre_ExpFastSweepDown";
                case 27: return "timbre_ExpSlowSweepDown";
                case 28: return "timbre_BulkSetLinearADSR";
                case 29: return "timbre_BulkSetExpADSR";
                case 30: return "timbre_Clear";
                case 31: return "noise_QuickSoftPuff";
                case 32: return "noise_SlowSoftPuff";
                case 33: return "noise_QuickHardPuff";
                case 34: return "noise_SlowHardPuff";
                case 35: return "noise_Clear";
                case 36: return "noise_co_apply_lpf_exp_fast";
                case 37: return "noise_co_apply_lpf_exp_slow";
                case 38: return "noise_co_BulkSetLinearADSR";
                case 39: return "noise_co_BulkSetExpADSR";
                case 40: return "noise_co_Clear";
                case 41: return "pitch_fast_linear_drop";
                case 42: return "pitch_slow_linear_drop";
                case 43: return "pitch_fast_linear_rise";
                case 44: return "pitch_slow_linear_rise";
                case 45: return "pitch_fast_exp_drop";
                case 46: return "pitch_slow_exp_drop";
                case 47: return "pitch_fast_exp_rise";
                case 48: return "pitch_slow_exp_rise";
                case 49: return "pitch_BulkSetLinearADSR";
                case 50: return "pitch_BulkSetExpADSR";
                case 51: return "pitch_Clear";
                case 52: return "sample_AttackOnLevel100";
                case 53: return "sample_AttackOnLevel75";
                case 54: return "sample_AttackOnLevel50";
                case 55: return "sample_AttackOnLevel25";
                case 56: return "sample_SustainOnLevel25";
                case 57: return "sample_ReleaseOnLevel25";
                case 58: return "sample_AttackOnLevel10";
                case 59: return "sample_SustainOnLevel10";
                case 60: return "sample_ReleaseOnLevel10";
                case 61: return "sample_AttackOnLevel5";
                case 62: return "sample_SustainOnLevel5";
                case 63: return "sample_ReleaseOnLevel5";
                case 64: return "sample_KS_100";
                case 65: return "sample_KS_75";
                case 66: return "sample_KS_50";
                case 67: return "sample_KS_25";
                case 68: return "sample_Clear";
                case 69: return "timbre_lfo_DeepAndSlow";
                case 70: return "timbre_lfo_ShallowAndSlow";
                case 71: return "timbre_lfo_DeepAndFast";
                case 72: return "timbre_lfo_ShallowAndFast";
                case 73: return "timbre_lfo_BulkSetLinearDSR";
                case 74: return "timbre_lfo_BulkSetExpDSR";
                case 75: return "timbre_lfo_Clear";

            }
            return "";
        }

        #endregion

        #region "============ METHODS - BIQUAD ============"

        public void CalcBiquadParams(ref biquad_eq_calculatedRow c, biquad_eqRow r)
        {
            if ((c == null) || (r == null)) { return; }

            if (r.q == 0) { r.q = (float)0.1; } //Added this to stop NaN from divide by zero

            Double norm;
            Double V = Math.Pow(10, Math.Abs(r.peak_gain) / 20);
            Double K = Math.Tan(Math.PI * r.fc / sample_rate);

            switch ((filter_types)r.filter_type_id)
            {
                case filter_types.lowpass:
                    norm = 1 / (1 + K / r.q + K * K);
                    c.a0 = (float)(K * K * norm);
                    c.a1 = 2 * c.a0;
                    c.a2 = c.a0;
                    c.b1 = (float)(2 * (K * K - 1) * norm);
                    c.b2 = (float)((1 - K / r.q + K * K) * norm);
                    break;

                case filter_types.highpass:
                    norm = 1 / (1 + K / r.q + K * K);
                    c.a0 = (float)(1 * norm);
                    c.a1 = -2 * c.a0;
                    c.a2 = c.a0;
                    c.b1 = (float)(2 * (K * K - 1) * norm);
                    c.b2 = (float)((1 - K / r.q + K * K) * norm);
                    break;

                case filter_types.bandpass:
                    norm = 1 / (1 + K / r.q + K * K);
                    c.a0 = (float)(K / r.q * norm);
                    c.a1 = 0;
                    c.a2 = -c.a0;
                    c.b1 = (float)(2 * (K * K - 1) * norm);
                    c.b2 = (float)((1 - K / r.q + K * K) * norm);
                    break;

                case filter_types.notch:
                    norm = 1 / (1 + K / r.q + K * K);
                    c.a0 = (float)((1 + K * K) * norm);
                    c.a1 = (float)(2 * (K * K - 1) * norm);
                    c.a2 = c.a0;
                    c.b1 = c.a1;
                    c.b2 = (float)((1 - K / r.q + K * K) * norm);
                    break;

                case filter_types.peak:
                    if (r.peak_gain >= 0)
                    {    // boost
                        norm = 1 / (1 + 1 / r.q * K + K * K);
                        c.a0 = (float)((1 + V / r.q * K + K * K) * norm);
                        c.a1 = (float)(2 * (K * K - 1) * norm);
                        c.a2 = (float)((1 - V / r.q * K + K * K) * norm);
                        c.b1 = c.a1;
                        c.b2 = (float)((1 - 1 / r.q * K + K * K) * norm);
                    }
                    else
                    {    // cut
                        norm = 1 / (1 + V / r.q * K + K * K);
                        c.a0 = (float)((1 + 1 / r.q * K + K * K) * norm);
                        c.a1 = (float)(2 * (K * K - 1) * norm);
                        c.a2 = (float)((1 - 1 / r.q * K + K * K) * norm);
                        c.b1 = c.a1;
                        c.b2 = (float)((1 - V / r.q * K + K * K) * norm);
                    }
                    break;
                case filter_types.lowshelf:
                    if (r.peak_gain >= 0)
                    {    // boost
                        norm = 1 / (1 + Math.Sqrt(2) * K + K * K);
                        c.a0 = (float)((1 + Math.Sqrt(2 * V) * K + V * K * K) * norm);
                        c.a1 = (float)(2 * (V * K * K - 1) * norm);
                        c.a2 = (float)((1 - Math.Sqrt(2 * V) * K + V * K * K) * norm);
                        c.b1 = (float)(2 * (K * K - 1) * norm);
                        c.b2 = (float)((1 - Math.Sqrt(2) * K + K * K) * norm);
                    }
                    else
                    {    // cut
                        norm = 1 / (1 + Math.Sqrt(2 * V) * K + V * K * K);
                        c.a0 = (float)((1 + Math.Sqrt(2) * K + K * K) * norm);
                        c.a1 = (float)(2 * (K * K - 1) * norm);
                        c.a2 = (float)((1 - Math.Sqrt(2) * K + K * K) * norm);
                        c.b1 = (float)(2 * (V * K * K - 1) * norm);
                        c.b2 = (float)((1 - Math.Sqrt(2 * V) * K + V * K * K) * norm);
                    }
                    break;
                case filter_types.highshelf:
                    if (r.peak_gain >= 0)
                    {    // boost
                        norm = 1 / (1 + Math.Sqrt(2) * K + K * K);
                        c.a0 = (float)((V + Math.Sqrt(2 * V) * K + K * K) * norm);
                        c.a1 = (float)(2 * (K * K - V) * norm);
                        c.a2 = (float)((V - Math.Sqrt(2 * V) * K + K * K) * norm);
                        c.b1 = (float)(2 * (K * K - 1) * norm);
                        c.b2 = (float)((1 - Math.Sqrt(2) * K + K * K) * norm);
                    }
                    else
                    {    // cut
                        norm = 1 / (V + Math.Sqrt(2 * V) * K + K * K);
                        c.a0 = (float)((1 + Math.Sqrt(2) * K + K * K) * norm);
                        c.a1 = (float)(2 * (K * K - 1) * norm);
                        c.a2 = (float)((1 - Math.Sqrt(2) * K + K * K) * norm);
                        c.b1 = (float)(2 * (K * K - V) * norm);
                        c.b2 = (float)((V - Math.Sqrt(2 * V) * K + K * K) * norm);
                    }
                    break;
            }

        }


        #endregion

    }


}
